#!/usr/bin/env python3

import argparse
from datetime import datetime
import json
import os
from pprint import pprint
import shutil
from typing import Dict

from jinja2 import Template


def render_template(file_path: str, variables: Dict):
    with open(file_path, 'r') as file:
        file_content = file.read()
        template = Template(file_content)
        return template.render(**variables)


def render_template_to_file(file_path: str, variables: Dict, output_file_path: str):
    with open(output_file_path, 'w') as output_file:
        output_file.write(render_template(file_path=file_path, variables=variables))


def print_file(purpose: str, file_path: str):
    print(f'Content of {purpose} file: {file_path}')
    print('>-----')
    with open(file_path, 'r') as file:
        print(file.read())
    print('<-----')


def prepare_file(source_path: str, output_path: str, name: str, extension: str, template_file: str, input_args: Dict):
    input_file = os.path.join(source_path, name + extension)
    custom_template_file = os.path.join(source_path, name + extension + '.j2')
    output_file = os.path.join(output_path, name + extension)
    if os.path.exists(input_file):
        print(f'Using existing {extension} file: {input_file}')
        shutil.copy(src=input_file, dst=output_file)
    elif os.path.exists(custom_template_file):
        print(f'Rendering {extension} file: {input_file} using custom template {custom_template_file}')
        render_template_to_file(custom_template_file, input_args, output_file)
    else:
        print(f'Rendering {extension} file: {input_file}')
        render_template_to_file(template_file, input_args, output_file)
    print_file(extension, output_file)    


def main():
    parser = argparse.ArgumentParser(description='Render config files for singularity build and VSC JupyterHub.')
    parser.add_argument('output', type=str, help='Output path')
    parser.add_argument('source', type=str, help='Path to the source files')
    parser.add_argument('template', type=str, help='Path to the template files')
    parser.add_argument('name', type=str, help='Common name of the data source files (.json, .meta, .def) - will also be used for output .sif file name')
    parser.add_argument('group', type=str, help='Output group name')
    parser.add_argument('docker_tag', type=str, help='Docker iamge tag to use')
    parser.add_argument('-t', '--testing', action='store_true', help='Add sysadmin to groups so its not visible for other users')
    args = parser.parse_args()
    
    def_file_template = os.path.join(args.template, 'singularity-template.def.j2')
    meta_file_template = os.path.join(args.template, 'meta-template.meta.j2')
    
    print(f'Generating config files using name: {args.name}')
    
    base_args = {
        'name': args.name,
        'group': args.group,
        'docker-tag': args.docker_tag,
        'generated-at': str(datetime.utcnow())
    }
    
    input_args = dict(base=base_args)
    
    # render input arguments if they are there
    var_args = {}
    var_file = os.path.join(args.source, args.name + '.json')
    if os.path.exists(var_file):
        print(f'Using var file: {var_file}')
        var_args = json.loads(render_template(var_file, input_args))
    else:
        print(f'No .json var file found - skipping. path={var_file}')

    if args.testing:
        print('Adding "sysadmin" group to required-groups for testing ...')
        required_groups = var_args.get('required-groups', [])
        required_groups.append('sysadmin')
        var_args['required-groups'] = required_groups

    input_args['var'] = var_args
    
    print('Using input args:')
    print('>-----')
    pprint(input_args)
    print('<-----')

    # DEF file
    prepare_file(args.source, args.output, args.name, '.def', def_file_template, input_args)

    # META file
    prepare_file(args.source, args.output, args.name, '.meta', meta_file_template, input_args)
    
    print('Config file generation done.')    


if __name__ == '__main__':
    main()
