#!/usr/bin/env bash

set -e

SCRIPT_DIR=$( cd -- "$( dirname -- "${BASH_SOURCE[0]}" )" &> /dev/null && pwd )

VSC_UNIFIED_IMAGE_PATH=/opt/sw/jupyterhub/notebooks/

function usage() {
    echo "usage: $0 <output> <source> <name> <group> <docker-tag> [dry-run] [testing]"
    echo ""
    echo "  output      temp output path where the files will be built"
    echo "  source      path to the json, def and meta source files"
    echo "  name        name of the input and output files e.g. all-spark.sif"
    echo "  group       output group directory"
    echo "  docker-tag  docker image tag to use"
    echo ""
    echo " optional:"
    echo "      dry-run     only render the config files and exit"
    echo "      testing     add testing flag for .def & .meta file generation"
    echo ""
}

if [[ $# -ne 5 ]] && [[ $# -ne 6 ]] && [[ $# -ne 7 ]] && [[ $# -ne 8 ]]; then
    usage
    exit 1
fi

OUTPUT_PATH=$( readlink -f $1 )
SOURCE_PATH=$( readlink -f $2 )
VARIANT_NAME="$3"
VARIANT_GROUP="$4"
VARIANT_DOCKER_TAG="$5"

PARAMETER="$6"
while [ ! -z "$PARAMETER" ]; do
    if [ "$PARAMETER" == 'dry-run' ]; then
        DRY_RUN=1
        echo "!!! Dry-run enabled !!!"
    elif [ "$PARAMETER" == 'testing' ]; then
        TESTING='--testing'
        echo "Testing enabled."
    elif [ ! -z "$PARAMETER" ]; then
        echo "Unknown parameter: $PARAMETER"
        echo ""
        usage
        exit 2
    fi
    shift
    PARAMETER="$6"
done

mkdir -p $OUTPUT_PATH

$SCRIPT_DIR/tools/generate_config.py $OUTPUT_PATH $SOURCE_PATH $SCRIPT_DIR/tools \
    $VARIANT_NAME $VARIANT_GROUP $VARIANT_DOCKER_TAG \
    $TESTING

VARIANT_IMAGE="$OUTPUT_PATH/${VARIANT_NAME}.sif"
VARIANT_DEF="$OUTPUT_PATH/${VARIANT_NAME}.def"
VARIANT_META="$OUTPUT_PATH/${VARIANT_NAME}.meta"

echo "Starting singularity build for $VARIANT_NAME: $VARIANT_IMAGE"
if [ -z "$DRY_RUN" ]; then
    singularity -v build $VARIANT_IMAGE $VARIANT_DEF
    # build with debug and verbosity flags
    # singularity -v -d build $VARIANT_IMAGE $VARIANT_DEF
else
    echo "singularity build $VARIANT_IMAGE $VARIANT_DEF"
fi

echo "Built files will be placed in subfolder: \"$VARIANT_GROUP\""

echo "Copying built image to VSC jupyterhub notebooks share: $VSC_UNIFIED_IMAGE_PATH"
if [ -z "$DRY_RUN" ]; then
    mkdir -p $VSC_UNIFIED_IMAGE_PATH/$VARIANT_GROUP/
    mv -v $VARIANT_IMAGE $VARIANT_DEF $VARIANT_META $VSC_UNIFIED_IMAGE_PATH/$VARIANT_GROUP/
else
    echo "mkdir -p $VSC_UNIFIED_IMAGE_PATH/$VARIANT_GROUP/"
    echo "mv -v $VARIANT_IMAGE $VARIANT_DEF $VARIANT_META $VSC_UNIFIED_IMAGE_PATH/$VARIANT_GROUP/"
fi

echo "Done $VARIANT_NAME."
echo ""
