-- SEQUENCES

CREATE SEQUENCE IF NOT EXISTS sq_analyst
    INCREMENT 1
    START 1;

CREATE SEQUENCE IF NOT EXISTS sq_owner
    INCREMENT 1
    START 1;

CREATE SEQUENCE IF NOT EXISTS sq_desktop
    INCREMENT 1
    START 1;

-- FUNCTIONS

CREATE OR REPLACE FUNCTION to_bigint(mac CHAR) RETURNS BIGINT AS
$$
BEGIN
    RETURN (('x' || lpad(REPLACE(mac, ':', ''), 16, '0'))::bit(64)::bigint);
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION to_mac(mac BIGINT) RETURNS CHAR AS
$$
BEGIN
    RETURN regexp_replace(to_hex(mac), '^(.{2})(.{2})(.{2})(.{2})(.{2})(.{2})',
                          '\1:\2:\3:\4:\5:\6');
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION from_ip(ip INET) RETURNS BIGINT AS
$$
BEGIN
    RETURN (ip - '0.0.0.0'::inet);
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION to_ip(ip BIGINT) RETURNS INET AS
$$
BEGIN
    RETURN ('0.0.0.0'::inet + ip);
END;
$$ LANGUAGE plpgsql;

-- IP ADDRESSES

CREATE OR REPLACE FUNCTION vpn_next_ip()
    RETURNS TABLE
            (
                ip INET
            )
AS
$$
DECLARE
    address INET;
BEGIN
    FOR i IN 0..253
        LOOP
            SELECT tmp.ip
            INTO address
            FROM (
                     SELECT to_ip(from_ip('10.8.0.1') + i) as ip
                     WHERE NOT EXISTS(SELECT u.vpn_ip FROM users u WHERE u.vpn_ip = to_ip(from_ip('10.8.0.1') + i) AND u.deleted_at IS NULL)
                 ) as tmp;
            IF address IS NOT NULL THEN
                RETURN QUERY SELECT address;
                EXIT;
            END IF;
        END LOOP;
END;
$$ LANGUAGE plpgsql;


-- MAC ADDRESSES

CREATE OR REPLACE FUNCTION default_mac(id INTEGER) RETURNS BIGINT AS
$$
BEGIN
    RETURN to_bigint('c2:fa:00:00:00:00') + id;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION default2_mac(id INTEGER) RETURNS BIGINT AS
$$
BEGIN
    RETURN to_bigint('c2:6d:00:00:00:00') + id;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION default3_mac(id INTEGER) RETURNS BIGINT AS
$$
BEGIN
    RETURN to_bigint('c2:11:00:00:00:00') + id;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION provider2_mac(id INTEGER) RETURNS BIGINT AS
$$
BEGIN
    RETURN to_bigint('c2:0c:00:00:00:00') + id;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION provider3_mac(id INTEGER) RETURNS BIGINT AS
$$
BEGIN
    RETURN to_bigint('c2:0e:00:00:00:00') + id;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION owner_mac(id INTEGER) RETURNS BIGINT AS
$$
BEGIN
    RETURN to_bigint('c2:4d:00:00:00:00') + id;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION analyst_mac(id INTEGER) RETURNS BIGINT AS
$$
BEGIN
    RETURN to_bigint('c2:3c:00:00:00:00') + id;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION analyst2_mac(id INTEGER) RETURNS BIGINT AS
$$
BEGIN
    RETURN to_bigint('c2:6c:00:00:00:00') + id;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION provider_mac(id INTEGER) RETURNS BIGINT AS
$$
BEGIN
    RETURN to_bigint('c2:0b:00:00:00:00') + id;
END;
$$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION vnc_mac(id INTEGER) RETURNS BIGINT AS
$$
BEGIN
    RETURN to_bigint('c2:8d:00:00:00:00') + id;
END;
$$ LANGUAGE plpgsql;

-- INSERT PROCEDURES

CREATE OR REPLACE PROCEDURE create_user(full_name CHAR, username CHAR, organization CHAR, vpn_password CHAR, email CHAR, tel CHAR)
    LANGUAGE plpgsql AS
$$
DECLARE
    ip_addr INET := vpn_next_ip();
BEGIN
    INSERT INTO users (full_name, username, organization, vpn_password, email, tel, vpn_ip)
    VALUES (full_name, username, organization, vpn_password, email, tel, ip_addr);
    COMMIT;
END;
$$;

CREATE OR REPLACE PROCEDURE create_owner_vm(username CHAR, password CHAR, hash CHAR, del INTEGER)
    LANGUAGE plpgsql AS
$$
BEGIN
    INSERT INTO owner_vms (username, init_pw, init_hash, expected_deletion)
    VALUES (username, password, hash, (SELECT NOW() + INTERVAL '100' day));
    COMMIT;
END;
$$;

CREATE OR REPLACE PROCEDURE create_analyst_vm(username CHAR, password CHAR, hash CHAR, del INTEGER)
    LANGUAGE plpgsql AS
$$
BEGIN
    INSERT INTO analyst_vms (username, init_pw, init_hash, expected_deletion)
    VALUES (username, password, hash, (SELECT NOW() + INTERVAL '100' day));
    COMMIT;
END;
$$;

CREATE OR REPLACE PROCEDURE create_desktop_vm(username CHAR, password CHAR, hash CHAR, vncpassword CHAR,
                                              vnchash CHAR, del INTEGER)
    LANGUAGE plpgsql AS
$$
BEGIN
    INSERT INTO desktop_vms (username, init_pw, init_hash, init_vncpw, init_vnchash, expected_deletion)
    VALUES (username, password, hash, vncpassword, vnchash, (SELECT NOW() + INTERVAL '100' day));
    COMMIT;
END;
$$;

-- TABLES

CREATE TABLE IF NOT EXISTS users
(
    username        character varying(255) not null primary key,
    full_name       character varying(255) not null,
    vpn_password    character varying(255) not null,
    vpn_ip          inet                   not null,
    organization    character varying(255) not null,
    email           character varying(255),
    tel             character varying(255),
    deleted_at      timestamp without time zone,
    created_at      timestamp without time zone default now()
);

CREATE TABLE IF NOT EXISTS analyst_vms
(
    id                integer primary key         default nextval('sq_analyst'),
    ip                inet,
    username          character varying(255) REFERENCES users (username),
    init_pw           character varying(255) not null,
    init_hash         character varying(255) not null,
    cur_hash          character varying(255),
    init_totp         character varying(255),
    cur_totp          character varying(255),
    pw_displayed      boolean not null default false,
    expected_deletion timestamp without time zone,
    deleted_at        timestamp without time zone,
    created_at        timestamp without time zone default now()
);

CREATE TABLE IF NOT EXISTS desktop_vms
(
    id                integer primary key         default nextval('sq_desktop'),
    ip                inet,
    username          character varying(255) REFERENCES users (username),
    init_pw           character varying(255) not null,
    init_hash         character varying(255) not null,
    cur_hash          character varying(255),
    init_totp         character varying(255),
    cur_totp          character varying(255),
    pw_displayed      boolean not null default false,
    init_vncpw        character varying(255) not null,
    init_vnchash      character varying(255) not null,
    cur_vnchash       character varying(255),
    expected_deletion timestamp without time zone,
    deleted_at        timestamp without time zone,
    created_at        timestamp without time zone default now()
);

CREATE TABLE IF NOT EXISTS owner_vms
(
    id                integer primary key         default nextval('sq_owner'),
    ip                inet,
    username          character varying(255) REFERENCES users (username),
    init_pw           character varying(255) not null,
    init_hash         character varying(255) not null,
    cur_hash          character varying(255),
    init_totp         character varying(255),
    cur_totp          character varying(255),
    pw_displayed      boolean not null default false,
    expected_deletion timestamp without time zone,
    deleted_at        timestamp without time zone,
    created_at        timestamp without time zone default now()
);

-- VIEWS

CREATE OR REPLACE VIEW analyst_vm_brief AS
SELECT id,
       concat('ossdipanalyst', id) as vm_name,
       concat('anaanalyst', id)    as internal_name,
       username
FROM analyst_vms
WHERE deleted_at IS NULL;

CREATE OR REPLACE VIEW desktop_vm_brief AS
SELECT id,
       concat('ossdipdesktop', id) as vm_name,
       concat('vncdesktop', id)    as internal_name,
       username
FROM desktop_vms
WHERE deleted_at IS NULL;

CREATE OR REPLACE VIEW desktop_vm_network AS
SELECT id,
       ip,
       to_mac(vnc_mac(id))       as mac,
       to_mac(analyst2_mac(id))  as analyst_mac,
       to_mac(provider2_mac(id)) as provider_mac,
       to_mac(default2_mac(id))  as default_mac
FROM desktop_vms
WHERE deleted_at IS NULL;

CREATE OR REPLACE VIEW analyst_vm_network AS
SELECT id,
       ip,
       to_mac(analyst_mac(id))  as mac,
       to_mac(default_mac(id))  as default_mac,
       to_mac(provider_mac(id)) as provider_mac
FROM analyst_vms
WHERE deleted_at IS NULL;

CREATE OR REPLACE VIEW owner_vm_brief AS
SELECT id,
       concat('ossdipowner', id) as vm_name,
       concat('ownowner', id)    as internal_name,
       username
FROM owner_vms
WHERE deleted_at IS NULL;

CREATE OR REPLACE VIEW owner_vm_network AS
SELECT id,
       ip,
       to_mac(owner_mac(id))    as mac,
       to_mac(default3_mac(id))  as default_mac,
       to_mac(provider3_mac(id)) as provider_mac
FROM owner_vms
WHERE deleted_at IS NULL;

CREATE OR REPLACE VIEW owner_vm_totp AS
SELECT v.id,
       coalesce(cur_hash, init_hash) as hash,
       coalesce(cur_totp, init_totp) as totp,
       concat('otpauth://totp/', v.username, '@', ip, '?secret=', coalesce(cur_totp, init_totp), '&issuer=',
              internal_name)         as qr
FROM owner_vms v
         JOIN owner_vm_brief b ON v.id = b.id
WHERE deleted_at IS NULL;

CREATE OR REPLACE VIEW analyst_vm_totp AS
SELECT v.id,
       coalesce(cur_hash, init_hash) as hash,
       coalesce(cur_totp, init_totp) as totp,
       concat('otpauth://totp/', v.username, '@', ip, '?secret=', coalesce(cur_totp, init_totp), '&issuer=',
              internal_name)         as qr
FROM analyst_vms v
         JOIN analyst_vm_brief b ON v.id = b.id
WHERE deleted_at IS NULL;

CREATE OR REPLACE VIEW desktop_vm_totp AS
SELECT v.id,
       init_vncpw                          as vnc_pw,
       coalesce(init_vnchash, cur_vnchash) as vnc_hash,
       concat('otpauth://totp/', v.username, '@', ip, '?secret=', coalesce(cur_totp, init_totp), '&issuer=',
              internal_name)         as qr
FROM desktop_vms v
         JOIN desktop_vm_brief b ON v.id = b.id
WHERE deleted_at IS NULL;